/*
 * Copyright 2020-2030, MateCloud, DAOTIANDI Technology Inc All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * Author: pangu(7333791@qq.com)
 */
package com.scnyw.trip.tcc.service.impl;

import cn.hutool.core.text.StrFormatter;
import com.scnyw.trip.tcc.entity.TccIdempotentEntity;
import com.scnyw.trip.tcc.mapper.TccIdempotentMapper;
import com.scnyw.trip.tcc.service.ITccIdempotentService;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Service;

/**
 * <p>
 * TCC 幂等控制表，用于记录 TCC 的 Try、Confirm 和 Cancel 操作与幂等控制，并作为防悬挂与空回滚的依据 服务实现类
 * </p>
 *
 * @author zhuangjiali
 * @since 2025-01-10
 */
@Service
public class TccIdempotentServiceImpl extends ServiceImpl<TccIdempotentMapper, TccIdempotentEntity> implements ITccIdempotentService {

    final String TRY_ACTION = "TRY";
    final String CONFIRM_ACTION = "CONFIRM";
    final String CANCEL_ACTION = "CANCEL";

    final String idempotent = "IDEMPOTENT";

    @Override
    public boolean idempotent(String xid) {
        return idempotentValidPass(idempotent, xid);
    }

    @Override
    public boolean tryOperate(String xid) {
        // 幂等控制
        if (idempotentValidPass(TRY_ACTION, xid)) {

            // 防悬挂判断-都不存在则正确，返回true
            boolean flag = !this.lambdaQuery()
                    .eq(TccIdempotentEntity::getXid, xid)
                    // confirmAction cancelAction
                    .in(TccIdempotentEntity::getAction, CONFIRM_ACTION, CANCEL_ACTION)
                    .exists();
            if (!flag) {
                log.warn(StrFormatter.format("防悬挂检测 未通过. xid: {}", xid));
            }
            return flag;
        }

        return false;
    }

    @Override
    public boolean confirmOperate(String xid) {
        if (idempotentValidPass(CONFIRM_ACTION, xid)) {
            // 空确认判断-存在则正确，返回true
            boolean flag = this.lambdaQuery()
                    // tryAction
                    .eq(TccIdempotentEntity::getXid, xid)
                    .eq(TccIdempotentEntity::getAction, TRY_ACTION)
                    .exists();
            if (!flag) {
                log.warn(StrFormatter.format("空确认检测 未通过. xid: {}", xid));
            }
            return flag;
        }
        return false;
    }

    @Override
    public boolean cancelOperate(String xid) {
        if (idempotentValidPass(CANCEL_ACTION, xid)) {
            // 空回滚判断-存在则正确，返回true
            boolean flag = this.lambdaQuery()
                    // tryAction
                    .eq(TccIdempotentEntity::getXid, xid)
                    .eq(TccIdempotentEntity::getAction, TRY_ACTION)
                    .exists();
            if (!flag) {
                log.warn(StrFormatter.format("空回滚检测 未通过. xid: {}", xid));
            }
            return flag;
        }
        return false;
    }

    private boolean idempotentValidPass(String action, String xid) {
        try {
            // 构造 TCC 操作记录
            TccIdempotentEntity tccIdempotentEntity = new TccIdempotentEntity();
            tccIdempotentEntity.setXid(xid);
            tccIdempotentEntity.setAction(action);
            tccIdempotentEntity.setRetryCount(0); // 初次插入，重试计数为 0
            tccIdempotentEntity.setErrorMessage(null); // 初次插入，无错误信息

            // 尝试保存记录
            this.save(tccIdempotentEntity);
        } catch (DuplicateKeyException e) {
            // 唯一索引异常：说明记录已存在
            log.warn(StrFormatter.format("Duplicate key detected for {}. xid: {}", action, xid));

            // 使用 lambdaUpdate 更新 retryCount 和 errorMessage
            this.lambdaUpdate()
                    .setSql("retry_count = retry_count + 1") // 重试次数 +1
                    .set(TccIdempotentEntity::getErrorMessage, "Duplicate key detected in TRY: " + e.getMessage())
                    .eq(TccIdempotentEntity::getXid, xid)
                    .eq(TccIdempotentEntity::getAction, action)
                    .update();

            return false;
        } catch (Exception e) {
            log.error("TCC TRY operation failed.", e);

            // 使用 lambdaUpdate 更新 retryCount 和 errorMessage
            this.lambdaUpdate()
                    .setSql("retry_count = retry_count + 1") // 重试次数 +1
                    .set(TccIdempotentEntity::getErrorMessage, "General exception: " + e.getMessage())
                    .eq(TccIdempotentEntity::getXid, xid)
                    .eq(TccIdempotentEntity::getAction, action)
                    .update();

            throw e; // 其他异常不应该拦截(如果上层也没拦截可能丢失此次重试更新记录)
        }

        return true; // 插入成功，Try 阶段处理完成
    }
}
